// BuildTwin Exact Match Interactive Features
class BuildTwinExactWebsite {
    constructor() {
        this.initializeComponents();
        this.setupInteractions();
        this.handleResponsiveLayout();
        this.addAccessibilityEnhancements();
    }

    initializeComponents() {
        // Cache DOM elements for better performance
        this.elements = {
            scheduleBtn: document.querySelector('.btx-cta-button'),
            locationCards: document.querySelectorAll('.btx-location-card'),
            categoryLinks: document.querySelectorAll('.btx-item-link'),
            getStartedLink: document.querySelector('.btx-start-link'),
            locationsContainer: document.querySelector('.btx-locations-container'),
            categoriesGrid: document.querySelector('.btx-categories-grid')
        };
    }

    setupInteractions() {
        // Schedule call button
        if (this.elements.scheduleBtn) {
            this.elements.scheduleBtn.addEventListener('click', this.handleScheduleCall.bind(this));
        }

        // Location cards interactions
        this.elements.locationCards.forEach(card => {
            card.addEventListener('click', this.handleLocationSelect.bind(this));
            card.addEventListener('mouseenter', this.handleLocationHover.bind(this));
            card.addEventListener('mouseleave', this.handleLocationLeave.bind(this));

            // Make cards focusable
            card.setAttribute('tabindex', '0');
            card.setAttribute('role', 'button');

            // Keyboard support
            card.addEventListener('keydown', (e) => {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    this.handleLocationSelect(e);
                }
            });
        });

        // Category links
        this.elements.categoryLinks.forEach(link => {
            link.addEventListener('click', this.handleCategoryClick.bind(this));
        });

        // Get started link
        if (this.elements.getStartedLink) {
            this.elements.getStartedLink.addEventListener('click', this.handleGetStarted.bind(this));
        }
    }


    handleResponsiveLayout() {
        let resizeTimer;

        window.addEventListener('resize', () => {
            clearTimeout(resizeTimer);
            resizeTimer = setTimeout(() => {
                this.adjustLayoutForViewport();
            }, 250);
        });

        // Initial layout adjustment
        this.adjustLayoutForViewport();
    }

    adjustLayoutForViewport() {
        const viewportWidth = window.innerWidth;

        // Dynamic adjustments based on exact breakpoints
        if (viewportWidth <= 480) {
            this.applyMobileLayout();
        } else if (viewportWidth <= 768) {
            this.applyTabletLayout();
        } else if (viewportWidth <= 1200) {
            this.applyDesktopSmallLayout();
        } else {
            this.applyDesktopLargeLayout();
        }
    }

    applyMobileLayout() {
        if (this.elements.locationsContainer) {
            this.elements.locationsContainer.style.gridTemplateColumns = '1fr 1fr';
        }
        if (this.elements.categoriesGrid) {
            this.elements.categoriesGrid.style.gridTemplateColumns = '1fr';
        }
    }

    applyTabletLayout() {
        if (this.elements.locationsContainer) {
            this.elements.locationsContainer.style.gridTemplateColumns = 'repeat(3, 1fr)';
        }
        if (this.elements.categoriesGrid) {
            this.elements.categoriesGrid.style.gridTemplateColumns = 'repeat(2, 1fr)';
        }
    }

    applyDesktopSmallLayout() {
        if (this.elements.locationsContainer) {
            this.elements.locationsContainer.style.gridTemplateColumns = 'repeat(3, 1fr)';
        }
        if (this.elements.categoriesGrid) {
            this.elements.categoriesGrid.style.gridTemplateColumns = 'repeat(3, 1fr)';
        }
    }

    applyDesktopLargeLayout() {
        if (this.elements.locationsContainer) {
            this.elements.locationsContainer.style.gridTemplateColumns = 'repeat(6, 1fr)';
        }
        if (this.elements.categoriesGrid) {
            this.elements.categoriesGrid.style.gridTemplateColumns = 'repeat(6, 1fr)';
        }
    }

    addAccessibilityEnhancements() {
        // Add ARIA labels to location cards
        this.elements.locationCards.forEach(card => {
            const locationName = card.querySelector('.btx-location-label').textContent;
            card.setAttribute('aria-label', `Select ${locationName} region`);
        });

        // Add skip navigation
        this.addSkipNavigation();

        // Announce dynamic content changes
        this.createAriaLiveRegion();
    }

    addSkipNavigation() {
        const skipLink = document.createElement('a');
        skipLink.href = '#main-content';
        skipLink.textContent = 'Skip to main content';
        skipLink.className = 'btx-skip-nav';

        // Skip link styles
        const skipStyles = `
            position: absolute;
            top: -40px;
            left: 8px;
            background: #1e3a5f;
            color: white;
            padding: 8px 16px;
            text-decoration: none;
            border-radius: 4px;
            z-index: 1000;
            font-size: 14px;
            transition: top 0.3s ease;
        `;

        skipLink.style.cssText = skipStyles;

        skipLink.addEventListener('focus', () => {
            skipLink.style.top = '8px';
        });

        skipLink.addEventListener('blur', () => {
            skipLink.style.top = '-40px';
        });

        document.body.insertBefore(skipLink, document.body.firstChild);

        // Add main content ID
        const mainContent = document.querySelector('.btx-content-area');
        if (mainContent) {
            mainContent.id = 'main-content';
        }
    }

    createAriaLiveRegion() {
        const liveRegion = document.createElement('div');
        liveRegion.setAttribute('aria-live', 'polite');
        liveRegion.setAttribute('aria-atomic', 'true');
        liveRegion.className = 'btx-sr-only';
        liveRegion.style.cssText = `
            position: absolute;
            width: 1px;
            height: 1px;
            padding: 0;
            margin: -1px;
            overflow: hidden;
            clip: rect(0, 0, 0, 0);
            white-space: nowrap;
            border: 0;
        `;

        document.body.appendChild(liveRegion);
        this.ariaLiveRegion = liveRegion;
    }

    showNotification(message, type = 'info') {
        // Update ARIA live region for screen readers
        if (this.ariaLiveRegion) {
            this.ariaLiveRegion.textContent = message;
        }

        // Visual notification (simple console log for demo)
        console.log(`${type.toUpperCase()}: ${message}`);

        // In a real implementation, this would show a toast notification
    }

    trackInteraction(eventName, eventData) {
        // Analytics tracking placeholder
        console.log(`Analytics: ${eventName}`, eventData);

        // Real implementation would send to analytics service
        // Example: gtag('event', eventName, eventData);
        // Example: analytics.track(eventName, eventData);
    }

    // Utility method for smooth scrolling
    smoothScrollTo(targetSelector) {
        const target = document.querySelector(targetSelector);
        if (target) {
            target.scrollIntoView({
                behavior: 'smooth',
                block: 'start'
            });
        }
    }
}

// Initialize when DOM is ready
document.addEventListener('DOMContentLoaded', () => {
    new BuildTwinExactWebsite();
});

// Performance monitoring
window.addEventListener('load', () => {
    if ('performance' in window) {
        const perfData = performance.getEntriesByType('navigation')[0];
        if (perfData) {
            console.log('Page load performance:', {
                loadTime: Math.round(perfData.loadEventEnd - perfData.loadEventStart),
                domContentLoaded: Math.round(perfData.domContentLoadedEventEnd - perfData.domContentLoadedEventStart),
                firstPaint: Math.round(perfData.responseEnd - perfData.requestStart)
            });
        }
    }
});
