<?php

namespace Egns\Helper;

use Egns_Helpers as GlobalEgns_Helpers;
use Elementor\Plugin;

if (!class_exists('Egns_Helper')) {

	/**
	 * Helper handlers class
	 */
	class Egns_Helper
	{

		/**
		 * Helper Class constructor
		 */
		function __construct()
		{
			// Before, After page load
			$this->actions();

			// Fire hook to include main header template
			add_action('egns_action_page_header_template', array($this, 'egns_load_page_header'));

			// Fire hook to include main footer template
			add_action('egns_action_page_footer_template', array($this, 'egns_load_page_footer'));
		}

		public function egns_load_page_header()
		{
			// Include header template
			echo apply_filters('egns_filter_header_template', self::egns_header_template());
		}

		public function egns_load_page_footer()
		{
			// Include footer template
			echo apply_filters('egns_filter_footer_template', self::egns_footer_template());
		}


		/**
		 * Method that echo module template part.
		 *
		 * @param string $module name of the module from inc folder
		 * @param string $template full path of the template to load
		 * @param string $slug
		 * @param array  $params array of parameters to pass to template
		 */
		public static function egns_template_part($module, $template, $slug = '', $params = array())
		{
			echo self::egns_get_template_part($module, $template, $slug, $params);
		}

		public static function get_project_post_option()
		{
			$posts = get_posts(['post_type' => 'project', 'posts_per_page' => -1]);
			$options = [];

			foreach ($posts as $post) {
				$options[$post->ID] = get_the_title($post->ID);
			}

			return $options;
		}

		public static function get_properties_post_option()
		{
			$posts = get_posts(['post_type' => 'properties', 'posts_per_page' => -1]);
			$options = [];

			foreach ($posts as $post) {
				$options[$post->ID] = get_the_title($post->ID);
			}

			return $options;
		}


		/**
		 * Method that load module template part.
		 *
		 * @param string $module name of the module from inc folder
		 * @param string $template full path of the template to load
		 * @param string $slug
		 * @param array  $params array of parameters to pass to template
		 *
		 * @return string - string containing html of template
		 */
		public static function egns_get_template_part($module, $template, $slug = '', $params = array())
		{

			//HTML Content from template
			$html          = '';
			$template_path = EGNS_INC_ROOT_DIR . '/' . $module;

			$temp = $template_path . '/' . $template;
			if (is_array($params) && count($params)) {
				extract($params);
			}

			$template = '';

			if (!empty($temp)) {
				if (!empty($slug)) {
					$template = "{$temp}-{$slug}.php";

					if (!file_exists($template)) {
						$template = $temp . '.php';
					}
				} else {
					$template = $temp . '.php';
				}
			}

			if ($template) {
				ob_start();
				include($template);
				$html = ob_get_clean();
			}

			return $html;
		}

		/**
		 * Method that check file exists or not.
		 *
		 * @param string $module name of the module from inc folder
		 * @param string $template full path of the template to load
		 * @param string $slug
		 *
		 * @return boolean - if exists then return true or false
		 */
		public static function egns_check_template_part($module, $template, $slug = '', $params = array())
		{

			//HTML Content from template
			$html          = '';
			$template_path = EGNS_INC_ROOT_DIR . '/' . $module;

			$temp = $template_path . '/' . $template;
			if (is_array($params) && count($params)) {
				extract($params);
			}

			$template = '';

			if (!empty($temp)) {
				if (!empty($slug)) {
					$template = "{$temp}-{$slug}.php";

					if (!file_exists($template)) {
						return false;
					} else {
						return true;
					}
				} else {
					$template = $temp . '.php';
					if (!file_exists($template)) {
						return false;
					} else {
						return true;
					}
				}
			}
		}

		/**
		 * Method that checks if forward plugin installed
		 *
		 * @param string $plugin - plugin name
		 *
		 * @return bool
		 */
		public static function egns_is_installed($plugin)
		{

			switch ($plugin) {
				case 'egns-core';
					return class_exists('Egns_Core');
					break;
				case 'woocommerce';
					return class_exists('WooCommerce');
					break;
				default:
					return false;
			}
		}

		public static function get_projects_post_option()
		{
			$posts = get_posts(['post_type' => 'project', 'posts_per_page' => -1]);
			$options = [];

			foreach ($posts as $post) {
				$options[$post->ID] = get_the_title($post->ID);
			}

			return $options;
		}


		/**
		 * Overwrite the theme option when page option is available.
		 *
		 * @param mixed theme option value.
		 * @param mixed page option value.
		 * @since   1.0.0
		 * @return bool 
		 */
		public static function is_enabled($theme_option, $page_option)
		{

			if (class_exists('CSF')) {

				if ($theme_option == 1) {

					if ($page_option == 1) {
						return true;
					} elseif (is_singular('product') || is_singular('portfolio') ||  is_singular('post') || is_single('post') || self::vernex_is_blog_pages() || is_404()) {
						return true;
					} elseif ($theme_option == 1 && empty($page_option) && $page_option != 0) {
						return true;
					} else {
						return false;
					}
				} else {
					return false;
				}
			} else {
				return true;
			}
		}
		/**
		 * Get all created menus with ID
		 *
		 * @since   1.0.0
		 */
		public static function list_all_menus()
		{
			// Get all registered menus
			$menus = get_terms('nav_menu', array('hide_empty' => true));

			// Initialize an empty array to store menu names with ID
			$menu_names = array();

			// Check if there are any menus
			if (!empty($menus)) {
				// Loop through each menu and add its name to the array
				foreach ($menus as $menu) {
					$menu_names[$menu->term_id] = $menu->name;
				}
			}

			// Return the array of menu names
			return $menu_names;
		}


		public static function display_related_posts_by_category($post_id)
		{
			$categories = wp_get_post_categories($post_id);

			if ($categories) {
				$args = array(
					'category__in'   => $categories,
					'post__not_in'   => array($post_id),
					'posts_per_page' => 4,                 // Number of related posts to display
					'orderby'        => 'rand'             // Random order
				);

				$related_posts = new \WP_Query($args);

				if ($related_posts->have_posts()) { ?>
					<div class="blog-post-area pt-90">
						<h6>You May Also Like</h6>
						<span class="line-break3"></span>
						<span class="line-break"></span>
						<div class="row gy-5">
							<?php while ($related_posts->have_posts()) {
								$related_posts->the_post();
							?>
								<div class="col-md-6">
									<div class="blog-card2 two">
										<?php
										self::egns_template_part('blog', 'templates/common/grid/thumbnail');
										self::egns_template_part('blog', 'templates/common/content');
										?>
									</div>
								</div>
							<?php
							}
							wp_reset_postdata();
							?>

						</div>
					</div>
				<?php
				}
			}
		}

		public static function  egns_project_value($key1, $key2 = '', $key3 = '', $default = '')
		{

			$page_options = get_post_meta(get_the_ID(), 'EGNS_PROJECT_META_ID', true);

			if (isset($page_options[$key1][$key2][$key3])) {
				return $page_options[$key1][$key2][$key3];
			} elseif (isset($page_options[$key1][$key2])) {
				return $page_options[$key1][$key2];
			} elseif (isset($page_options[$key1])) {
				return $page_options[$key1];
			} else {
				return $default;
			}
		}
		public static function  egns_properties_value($key1, $key2 = '', $key3 = '', $default = '')
		{

			$page_options = get_post_meta(get_the_ID(), 'EGNS_PROPERTIES_META_ID', true);

			if (isset($page_options[$key1][$key2][$key3])) {
				return $page_options[$key1][$key2][$key3];
			} elseif (isset($page_options[$key1][$key2])) {
				return $page_options[$key1][$key2];
			} elseif (isset($page_options[$key1])) {
				return $page_options[$key1];
			} else {
				return $default;
			}
		}


		/**
		 * clean special chars, spaces with hyphens
		 *
		 * @since   1.0.0
		 */
		public static function clean($string)
		{
			$string = str_replace(' ', '', $string);                  // Replaces all spaces with hyphens.
			$string = preg_replace('/[^A-Za-z0-9\-]/', '', $string);  // Removes special chars.

			return preg_replace('/-+/', '', $string);  // Replaces multiple hyphens with single one.
		}

		/**
		 * Get first category with link
		 *
		 * @since   1.0.0
		 */
		public static function the_first_category()
		{
			$categories = get_the_category();
			if (!empty($categories)) {
				echo '<a href="' . esc_url(get_category_link($categories[0]->term_id)) . '">' . esc_html($categories[0]->name) . '</a>';
			}
		}


		/**
		 * Option Dynamic Styles (Header)
		 *
		 * @since   1.0.0
		 */
		public function egns_header_template()
		{
			$egns_scrollTop = self::egns_get_theme_option('scrollTop_enable');

			if (1 == $egns_scrollTop) {
				get_template_part('inc/common/templates/scroll_top'); // scrollTop Button
			}

			$get_header_style 	  	 	= self::egns_get_theme_option('header_menu_style');
			$get_page_header_style 		= self::egns_page_option_value('page_header_menu_style');
			$page_main_header_enable 	= self::egns_page_option_value('page_main_header_enable');

			// Page Header Layout
			if (!empty($page_main_header_enable) && ($page_main_header_enable == 'disable') && class_exists('CSF')) {
				$get_header_style = 'no_header';
			} else {
				if (!empty($get_page_header_style) && $get_page_header_style == 'header_one' && class_exists('CSF')) {
					$get_header_style = 'header_one';
				}
				if (!empty($get_page_header_style) && $get_page_header_style == 'header_two' && class_exists('CSF')) {
					$get_header_style = 'header_two';
				}
				if (!empty($get_page_header_style) && $get_page_header_style == 'header_three' && class_exists('CSF')) {
					$get_header_style = 'header_three';
				}
				if (!empty($get_page_header_style) && $get_page_header_style == 'header_four' && class_exists('CSF')) {
					$get_header_style = 'header_four';
				}
				if (!empty($get_page_header_style) && $get_page_header_style == 'header_five' && class_exists('CSF')) {
					$get_header_style = 'header_five';
				}
				if (!empty($get_page_header_style) && $get_page_header_style == 'header_six' && class_exists('CSF')) {
					$get_header_style = 'header_six';
				}
			}

			switch ($get_header_style) {
				case 'header_one':
					get_template_part('inc/header/templates/parts/header_one');
					break;
				case 'header_two':
					get_template_part('inc/header/templates/parts/header_two');
					break;
				case 'header_three':
					get_template_part('inc/header/templates/parts/header_three');
					break;
				case 'header_four':
					get_template_part('inc/header/templates/parts/header_four');
					break;
				case 'header_five':
					get_template_part('inc/header/templates/parts/header_five');
					break;
				case 'header_six':
					get_template_part('inc/header/templates/parts/header_six');
					break;
				case 'no_header':
					break;
				default:
					get_template_part('inc/header/templates/parts/header_one');
					break;
			}
		}




		/**
		 * Option Dynamic Styles (Footer)
		 *
		 * @since   1.0.0
		 */
		public function egns_footer_template()
		{
			$page_main_footer_enable = self::egns_page_option_value('page_main_footer_enable');

			$imported_demo = get_option('imported_demo');

			if ($imported_demo && is_array($imported_demo)) {
				// Use $demo_name as needed
				$demo_name = $imported_demo['import_file_name'];
			} else {
				// Handle the case where the option does not exist or is not an array.
				$demo_name = '';
			}

			// Page footer Layout
			if (!empty($page_main_footer_enable) && ($page_main_footer_enable == 'disable') && class_exists('CSF')) {
				$demo_name = 'no_footer';
			}

			switch ($demo_name) {
				case 'Main':
					get_template_part('inc/footer/templates/parts/footer_one');
					break;
				case 'Car':
					get_template_part('inc/footer/templates/parts/footer_two');
					break;
				case 'Antiques':
					get_template_part('inc/footer/templates/parts/footer_three');
					break;
				case 'Art':
					get_template_part('inc/footer/templates/parts/footer_four');
					break;
				case 'Gadget-and-Technology':
					get_template_part('inc/footer/templates/parts/footer_five');
					break;
				case 'Book-and-Comic':
					get_template_part('inc/footer/templates/parts/footer_six');
					break;
				case 'Multipurpose':
					get_template_part('inc/footer/templates/parts/footer_seven');
					break;
				case 'no_footer':
					break;
				default:
					get_template_part('inc/footer/templates/parts/footer_one');
					break;
			}
		}


		/**
		 * Is Pages
		 *
		 * @since   1.0.0
		 */
		public static function egns_is_blog_pages()
		{
			return ((((is_search()) || (is_404()) || is_archive()) || (is_single()) || (is_singular())  ||  (is_author()) || (is_category()) || (is_home()) || (is_tag()))) ? true : false;
		}

		/**
		 * Is Blog Pages
		 *
		 * @since   1.2.0
		 */
		public static function vernex_is_blog_pages()
		{
			return ((((is_search()) || is_archive()) ||  (is_author()) || (is_category()) || (is_home())  || (is_tag()))) ? true : false;
		}

		/**
		 * Get theme options.
		 *
		 * @param string $opts Required. Option name.
		 * @param string $key Required. Option key.
		 * @param string $default Optional. Default value.
		 * @since   1.0.0
		 */

		public static function egns_get_theme_option($key, $key2 = '', $default = '')
		{
			$egns_theme_options = get_option('egns_theme_options');

			if (!empty($key2)) {
				return isset($egns_theme_options[$key][$key2]) ? $egns_theme_options[$key][$key2] : $default;
			} else {
				return isset($egns_theme_options[$key]) ? $egns_theme_options[$key] : $default;
			}
		}

		/**
		 * Css Minifier.
		 * @param $css get css
		 * @since   1.0.0
		 */
		public static function cssminifier($css)
		{
			$css = str_replace(
				["\r\n", "\r", "\n", "\t", '    '],
				'',
				preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', trim($css))
			);
			return str_replace(
				['{ ', ' }', ' {', '} ', ' screen and '],
				['{', '}', '{', '}', ''],
				$css
			);
		}

		/**
		 * Return Page Option Value Based on Given Page Option ID.
		 *
		 * @since 1.0.0
		 *
		 * @param string $page_option_key Optional. Page Option id. By Default It will return all value.
		 * 
		 * @return mixed Page Option Value.
		 */
		public static function  egns_page_option_value($key1, $key2 = '', $default = '')
		{

			$page_options = get_post_meta(get_the_ID(), 'egns_page_options', true);

			if (isset($page_options[$key1][$key2])) {

				return $page_options[$key1][$key2];
			} else {
				if (isset($page_options[$key1])) {

					return  $page_options[$key1];
				} else {
					return $default;
				}
			}
		}

		/**
		 * Get Blog layout
		 *
		 * @since   1.0.0
		 */
		public static function egns_post_layout()
		{
			$egns_theme_options = get_option('egns_theme_options');

			$blog_layout = !empty($egns_theme_options['blog_layout_options']) ? $egns_theme_options['blog_layout_options'] : 'default';

			return $blog_layout;
		}

		/**
		 * Escape any String with Translation
		 *
		 * @since   1.0.0
		 */

		public static function egns_translate($value)
		{
			echo sprintf(__('%s', 'vernex'), $value);
		}
		/**
		 * Escape any String with Translation
		 *
		 * @since   1.0.0
		 */

		public static function egns_translate_with_escape_($value)
		{
			$value = esc_html($value);
			echo sprintf(__('%s', 'vernex'), $value);
		}

		/**
		 * Dynamic blog layout for post archive pages.
		 *
		 * @since   1.0.0
		 */
		public static function egns_dynamic_blog_layout()
		{
			$blog_layout = self::egns_post_layout();
			if (!empty($blog_layout)) {
				if ('default' == $blog_layout) {
					get_template_part('template-parts/blog/blog-standard');
				} elseif ($blog_layout == 'layout-01') {
					get_template_part('template-parts/blog/blog-grid-sidebar');
				}
			} else {
				get_template_part('template-parts/blog/blog-standard');
			}
		}

		/**
		 * 
		 * @return [string] audio url for audio post
		 */
		public static function egns_embeded_audio($width, $height)
		{
			$url = esc_url(get_post_meta(get_the_ID(), 'egns_audio_url', 1));
			if (!empty($url)) {
				return '<div class="post-media">' . wp_oembed_get($url, array('width' => $width, 'height' => $height)) . '</div>';
			}
		}

		/**
		 * @return [string] Checks For Embed Audio In The Post.
		 */
		public static function egns_has_embeded_audio()
		{
			$url = esc_url(get_post_meta(get_the_ID(), 'egns_audio_url', 1));
			if (!empty($url)) {
				return true;
			} else {
				return false;
			}
		}

		/**
		 * Post Meta Box Key Information
		 *
		 * @param  String $meta_key
		 */

		public static function egns_post_meta_box_value($meta_key, $meta_key_value)
		{
			return get_post_meta(get_the_ID(), $meta_key, true)[$meta_key_value];
		}

		/**
		 * Find Related Project
		 *
		 * @param  int $post_id
		 * @param  String $post_type
		 * @param  String $custom_post_taxonomy
		 * 
		 */

		public static function egns_find_related_project($post_id, $post_type, $custom_post_taxonomy)
		{
			//get the taxonomy terms of custom post type
			$customTaxonomyTerms = wp_get_object_terms($post_id, $custom_post_taxonomy, array('fields' => 'ids'));

			//query arguments
			$args = array(
				'post_type'      => $post_type,
				'post_status'    => 'publish',
				'posts_per_page' => 5,
				'orderby'        => 'rand',
				'tax_query'      => array(
					array(
						'taxonomy' => $custom_post_taxonomy,
						'field'    => 'id',
						'terms'    => $customTaxonomyTerms
					)
				),
				'post__not_in' => array($post_id),
			);
			return $args;
		}


		/**
		 * @return [string] Embed gallery for the post.
		 */
		public static function egns_gallery_images()
		{
			$images = get_post_meta(get_the_ID(), 'egns_gallery_images', 1);

			$images = explode(',', $images);
			if ($images && count($images) > 1) {
				$gallery_slide  = '<div class="swiper blog-archive-slider">';
				$gallery_slide .= '<div class="swiper-wrapper">';
				foreach ($images as $image) {
					$gallery_slide .= '<div class="swiper-slide"><a href="' . get_the_permalink() . '"><img src="' . wp_get_attachment_image_url($image, 'full') . '" alt="' . esc_attr(get_the_title()) . '"></a></div>';
				}
				$gallery_slide .= '</div>';
				$gallery_slide .= '</div>';

				$gallery_slide .= '<div class="slider-arrows arrows-style-2 sibling-3 text-center d-flex flex-row justify-content-between align-items-center w-100">';
				$gallery_slide .= '<div class="blog1-prev swiper-prev-arrow" tabindex="0" role="button" aria-label="' . esc_html('Previous slide') . '"> <i class="bi bi-arrow-left"></i> </div>';

				$gallery_slide .= '<div class="blog1-next swiper-next-arrow" tabindex="0" role="button" aria-label="' . esc_html('Next slide') . '"><i class="bi bi-arrow-right"></i></div>';
				$gallery_slide .= '</div>';

				return $gallery_slide;
			}
		}

		/**
		 * @return [string] Has Gallery for Gallery post.
		 */
		public static function has_egns_gallery()
		{
			$images = get_post_meta(get_the_ID(), 'egns_gallery_images', 1);
			if (!empty($images)) {
				return true;
			} else {
				return false;
			}
		}

		/**
		 * @return string get the attachment image.
		 */
		public static function egns_thumb_image()
		{
			$image = get_post_meta(get_the_ID(), 'egns_thumb_images', 1);
			echo '<a href="' . get_the_permalink() . '"><img src="' . isset($image['url']) . '" alt="' . esc_attr(get_the_title()) . ' "class="img-fluid wp-post-image"></a>';
		}

		/**
		 * @return [quote] text for quote post
		 */
		public static function egns_quote_content()
		{
			$text = get_post_meta(get_the_ID(), 'egns_quote_text', 1);
			if (!empty($text)) {
				return sprintf(esc_attr__("%s", 'vernex'), $text);
			}
		}

		/**
		 * @return [string] video url for video post
		 */
		public static function egns_embeded_video($width = '', $height = '')
		{
			$url = esc_url(get_post_meta(get_the_ID(), 'egns_video_url', 1));
			if (!empty($url)) {
				return wp_oembed_get($url, array('width' => $width, 'height' => $height));
			}
		}

		/**
		 * @return [string] Has embed video for video post.
		 */
		public static function has_egns_embeded_video()
		{
			$url = esc_url(get_post_meta(get_the_ID(), 'egns_video_url', 1));
			if (!empty($url)) {
				return true;
			} else {
				return false;
			}
		}

		/**
		 * Audio layout for post formet audio.
		 *
		 * @since   1.0.0
		 */
		public static function dynamic_audio_layout()
		{

			$blog_layout = self::egns_post_layout();
			$layout      = self::egns_embeded_audio(400, 275);
			if (is_single()) {
				$layout = self::egns_embeded_audio(850, 550);
			} elseif ($blog_layout == 'layout-02') {
				$layout = self::egns_embeded_audio(500, 450);
			} elseif ($blog_layout == 'layout-01') {
				$layout = self::egns_embeded_audio(400, 275);
			} else {
				$layout = self::egns_embeded_audio(400, 275);
			}
			return $layout;
		}

		/**
		 * Video layout for post formet video.
		 *
		 * @since   1.0.0
		 */
		public static function dynamic_video_layout()
		{

			$blog_layout = self::egns_post_layout();
			$layout      = self::egns_embeded_video(400, 275);
			if (is_single()) {
				$layout = self::egns_embeded_video(1050, 474);
			} elseif ($blog_layout == 'layout-02') {
				$layout = self::egns_embeded_video(600, 550);
			} elseif ($blog_layout == 'layout-01') {
				$layout = self::egns_embeded_video(400, 275);
			} else {
				$layout = self::egns_embeded_video(400, 275);
			}
			return $layout;
		}

		public static function get_theme_logo($logo_url, $echo = true)
		{
			if (has_custom_logo()):
				the_custom_logo();

			elseif (!empty($logo_url)):
				?>
				<?php if (!empty($logo_url)): ?>
					<a href="<?php echo esc_url(home_url('/')); ?>">
						<img class="img-fluid" src="<?php echo esc_url($logo_url); ?>" alt="<?php echo esc_attr(get_bloginfo('name', 'display')); ?>">
					</a>
				<?php endif ?>

				<?php
			else : {
				?>
					<div class="site-title">
						<h3><a href="<?php echo esc_url(home_url('/')) ?>"><?php echo esc_html(get_bloginfo('name')); ?></a></h3>
					</div>

				<?php
				}
			endif;
		}


		public static function get_copyright_theme_logo($logo_url, $echo = true)
		{
			if (has_custom_logo()):
				the_custom_logo();
			elseif (!empty($logo_url)):
				?>
				<a href="<?php echo esc_url(home_url('/')); ?>" title="<?php echo esc_attr(get_bloginfo('name', 'display')); ?>">
					<?php if (!empty($logo_url)): ?>
						<img class="img-fluid" src="<?php echo esc_url($logo_url); ?>" alt="<?php echo esc_attr(get_bloginfo('name', 'display')); ?>"></a>
			<?php endif ?>
			<?php
			endif;
		}


		/**
		 * Menu links.
		 *
		 * @param   string $theme_location menu type.
		 * @param   string $container_class main class.
		 * @param   string $after icon tag.
		 * @param   string $menu_class .
		 * @param   string $depth.
		 * @since   1.0.0
		 */
		public static function egns_get_theme_menu($theme_location = 'primary-menu', $container_class = '', $after = '<i class="bi bi-plus dropdown-icon"></i>', $menu_class = 'menu-list', $depth = 3, $echo = true)
		{
			if (has_nav_menu('primary-menu')) {
				wp_nav_menu(
					array(
						'theme_location'  => $theme_location,
						'container'       => false,              // This will prevent any container div from being added
						'container_class' => $container_class,
						'link_before'     => '',
						'link_after'      => '',
						'after'           => $after,
						'container_id'    => '',
						'menu_class'      => $menu_class,
						'fallback_cb'     => '',
						'menu_id'         => '',
						'depth'           => $depth,
					)
				);
			} else {
				if (is_user_logged_in()) { ?>
				<div class="set-menu">
					<h4>
						<a href="<?php echo admin_url(); ?>nav-menus.php">
							<?php echo esc_html('Set Menu Here...', 'vernex'); ?>
						</a>
					</h4>
				</div>
			<?php }
			}
		}


		/**

		 * Displays SVG content.

		 * This function retrieves SVG content from a file URL and displays it. If a filesystem object

		 * is provided, it uses it to fetch the file contents. Otherwise, it uses WordPress functions

		 * to fetch the contents remotely. The SVG content is then echoed directly.

		 * @since 1.0.0

		 * @param string $file_url The URL of the SVG file.

		 * @param object $filesystem Optional. The filesystem object. Defaults to null.
		 */

		public static function display_svg($file_url, $filesystem = null)
		{
			if (is_null($filesystem) && function_exists('WP_Filesystem')) {
				global $wp_filesystem;
				$filesystem = $wp_filesystem;
			} elseif (is_null($filesystem)) {
				include_once ABSPATH . 'wp-admin/includes/file.php';
			}

			$file_contents = '';
			if ($filesystem) {
				$file_contents = $filesystem->get_contents($file_url);
			} else {
				$response = wp_remote_get($file_url);
				if (!is_wp_error($response) && $response['response']['code'] === 200) {
					$file_contents = wp_remote_retrieve_body($response);
				}
			}

			if (!empty($file_contents)) {
				echo sprintf('%s', $file_contents);
			}
		}

		/**
		 * Post Details Pagination
		 * @since   1.0.0
		 */

		public static function egns_get_post_pagination()
		{
			wp_link_pages(
				array(
					'before'         => '<ul class="pagination d-flex justify-content-center align-items-center"><span 		class="page-title">' . esc_html__('Pages: ', 'vernex') . '</span><li>',
					'after'          => '</li></ul>',
					'link_before'    => '',
					'link_after'     => '',
					'next_or_number' => 'number',
					'separator'      => '</li><li>',
					'pagelink'       => '%',
					'echo'           => 1
				)
			);
		}

		/**
		 * Option Dynamic Styles.
		 *
		 * @since   1.0.0
		 */
		public function egns_enqueue_scripts()
		{
			$objects = array(
				'sticky_header_enable' => self::sticky_header_enable(),
				'animation_enable'     => self::animation_enable(),
				'is_egns_core_enable'  => class_exists('CSF') ? true : false,
			);
			wp_localize_script('egns-main', 'theme_options', $objects);
		}

		public static function sticky_header_enable()
		{
			$page_sticky_option = Egns_Helper::egns_page_option_value('sticky_header_enable');
			$sticky_header      = Egns_Helper::egns_get_theme_option('header_sticky_enable');

			if (Egns_Helper::is_enabled($sticky_header, $page_sticky_option)) {
				return true;
			} else {
				return false;
			}
		}

		public static function animation_enable()
		{
			$animation_enable = Egns_Helper::egns_get_theme_option('animation_enable');

			if ($animation_enable == 1) {
				return true;
			} else {
				return false;
			}
		}

		/**
		 * Get Page Options Value
		 *
		 * @since   1.0.0
		 */

		public static function egns_get_options_value($theme_option, $page_option)
		{
			if (!empty($page_option)) {
				return $page_option;
			} else {
				return $theme_option;
			}
		}



		/**
		 * Post layout for post formet.
		 *
		 * @since   1.0.0
		 */
		public static function dynamic_post_format()
		{
			$format = get_post_format(get_the_ID());

			switch ($format) {
				case 'video';
					self::egns_template_part('post-thumb', 'video');
					break;
				case 'audio';
					self::egns_template_part('post-thumb', 'audio');
					break;
				case 'gallery';
					self::egns_template_part('post-thumb', 'gallery');
					break;
				case 'quote';
					self::egns_template_part('post-thumb', 'quote');
					break;
				case 'image';
					self::egns_template_part('post-thumb', 'image');
					break;
				default:
					break;
			}
		}


		/**
		 * Define the core functionality of the.
		 *
		 * @since   1.0.0
		 */
		public function actions()
		{
			add_action('egns_page_before', array($this, 'open_container'));
			add_action('egns_page_after', array($this, 'close_container'));
			add_action('egns_post_before', array($this, 'post_open_container'));
			add_action('egns_post_after', array($this, 'post_close_container'));
			add_action('egns_header_template', [$this, 'egns_header_template']);
		}


		/**
		 * Is elementor.
		 *
		 * @since   1.0.0
		 */
		public static function is_elementor()
		{
			if (self::vernex_is_blog_pages()) {
				return false;
			}
			if (did_action('elementor/loaded')) {
				return Plugin::$instance->documents->get(get_the_ID())->is_built_with_elementor();
			} else {
				return false;
			}
		}

		/**
		 * Open Page Container.
		 *
		 * @since   1.0.0
		 */
		public function open_container()
		{
			if (!self::is_elementor()): ?>
			<div class="container">
			<?php
			endif;
		}

		/**
		 * Close Page Container.
		 *
		 * @since   1.0.0
		 */
		public function close_container()
		{
			if (!self::is_elementor()):
			?>
			</div> <!-- End Main Content Area  -->
		<?php endif;
		}

		/**
		 * Post Open Container.
		 *
		 * @since   1.0.0
		 */
		public function post_open_container()
		{
			if (is_single()) {
		?>
			<div class="blog-details">
			<?php
			} else {
			?>
				<div>
				<?php
			}
		}

		/**
		 * Post Close Container.
		 *
		 * @since   1.0.0
		 */
		public function post_close_container()
		{
				?>
				</div>
	<?php
		}
	} // end Main Egns Helper class






}
