<?php

namespace Egns_Core;

/**
 * All Elementor widget init
 * 
 * @since 1.0.0
 */

if (!defined('ABSPATH')) {
	exit();  // exit if access directly
}

if (!class_exists('Egns_Helper')) {

	class Egns_Helper
	{


		/**
		 * Helper Class constructor
		 */
		function __construct()
		{
			// call only public function here 
		}



		/**
		 * Get all created menus with ID
		 *
		 * @since   1.0.0
		 */
		public static function list_all_menus()
		{
			// Get all registered menus
			$menus = get_terms('nav_menu', array('hide_empty' => true));

			// Initialize an empty array to store menu names with ID
			$menu_names = array();

			// Check if there are any menus
			if (!empty($menus)) {
				// Loop through each menu and add its name to the array
				foreach ($menus as $menu) {
					$menu_names[$menu->slug] = $menu->name;
				}
			}

			// Return the array of menu names
			return $menu_names;
		}

		public static function  egns_properties_value($key1, $key2 = '', $key3 = '', $default = '')
		{

			$page_options = get_post_meta(get_the_ID(), 'EGNS_PROPERTIES_META_ID', true);

			if (isset($page_options[$key1][$key2][$key3])) {
				return $page_options[$key1][$key2][$key3];
			} elseif (isset($page_options[$key1][$key2])) {
				return $page_options[$key1][$key2];
			} elseif (isset($page_options[$key1])) {
				return $page_options[$key1];
			} else {
				return $default;
			}
		}


		public static function get_properties_types_for_select()
		{
			$taxonomy = 'properties-type';

			// Get terms for the specified custom post type and taxonomy
			$terms = get_terms([
				'taxonomy' => $taxonomy,
				'hide_empty' => true,
			]);

			$options = [];

			if ($terms && !is_wp_error($terms)) {
				foreach ($terms as $term) {
					$options[$term->term_id] = $term->name;
				}
			}

			return $options;
		}


		public static function get_post_list_by_post_type($post_type)
		{
			$return_val = [];
			$args       = array(
				'post_type'      => $post_type,
				'posts_per_page' => -1,
				'post_status'    => 'publish',

			);
			$all_post = new \WP_Query($args);

			while ($all_post->have_posts()) {
				$all_post->the_post();
				$return_val[get_the_ID()] = get_the_title();
			}
			wp_reset_query();
			return $return_val;
		}

		public static function get_all_post_key($post_type)
		{
			$return_val = [];
			$args       = array(
				'post_type'      => $post_type,
				'posts_per_page' => 6,
				'post_status'    => 'publish',

			);
			$all_post = new \WP_Query($args);

			while ($all_post->have_posts()) {
				$all_post->the_post();
				$return_val[] = get_the_ID();
			}
			wp_reset_query();
			return $return_val;
		}

	
		/**
		 * Get WooCommerce product categories
		 *
		 * @return array|WP_Error
		 */
		public static function get_woocommerce_product_categories()
		{
			// Get product categories
			$product_categories = get_terms('product_cat', array('hide_empty' => true));

			// Check if there was an error retrieving categories
			if (is_wp_error($product_categories)) {
				// Handle the error (you might log it, display a message, or take other actions)
				return $product_categories;
			}

			// Check if there are no categories
			if (empty($product_categories)) {
				// Handle the case where there are no categories (return a default value, show a message, etc.)
				return [];  // or return some default value
			}

			// Initialize an empty array to store category options
			$category_options = [];

			// Loop through each category
			foreach ($product_categories as $category) {
				// Build the associative array with term ID as key and category name as value
				$category_options[$category->term_id] = $category->name;
			}

			// Return the category options array
			return $category_options;
		}

		public static function get_blog_categories()
		{
			$categories       = get_categories();  // Get all categories.
			$category_options = [];
			foreach ($categories as $category) {
				$category_options[$category->term_id] = $category->name;
			}

			return $category_options;
		}


		/**
		 * filtering posts by title
		 *
		 * @return void
		 */
		public static function get_blog_post_options()
		{
			$posts   = get_posts(['post_type' => 'post', 'posts_per_page' => -1]);
			$options = [];

			foreach ($posts as $post) {
				$options[$post->ID] = get_the_title($post->ID);
			}

			return $options;
		}



		/**
		 * Filtering posts by title and multiple categories
		 *
		 * @param array|null $selected_category_ids Optional. Array of Category IDs to filter posts. Default is null (no filter).
		 * @return array Associative array of post IDs and their titles.
		 */
		public static function get_blog_post_cat_options($selected_category_ids = null)
		{
			// Initialize query arguments
			$query_args = [
				'post_type'      => 'post',
				'posts_per_page' => -1,
			];

			// If category IDs are provided, add them to the query arguments
			if (!empty($selected_category_ids)) {
				$query_args['category__in'] = $selected_category_ids;
			}

			// Get posts based on the query arguments
			$posts   = get_posts($query_args);
			$options = [];

			// Loop through posts and build options array
			foreach ($posts as $post) {
				$options[$post->ID] = get_the_title($post->ID);
			}

			return $options;
		}

		/*
		* Fetch Project Categories
		*/
		public static function get_project_categories()
		{
			$taxonomy = 'project-category';

			
			$terms = get_terms([
				'taxonomy'   => $taxonomy,
				'hide_empty' => true, 
			]);

			$category_options = [];

			if (!is_wp_error($terms)) {
				foreach ($terms as $term) {
					$category_options[$term->term_id] = $term->name;
				}
			}

			return $category_options;
		}


		/**
		 * Filtering Projects by title and categories
		 *
		 * @param array|null $selected_category_ids Optional. Array of Category IDs to filter projects. Default is null (no filter).
		 * @return array Associative array of project IDs and their titles.
		 */

		public static function get_project_options($selected_category_ids = null)
		{
			$query_args = [
				'post_type'      => 'project',
				'posts_per_page' => -1,
				'post_status'    => 'publish', 
			];

			// If category IDs are provided, add them to the query arguments
			if (!empty($selected_category_ids)) {
				$query_args['tax_query'] = [
					[
						'taxonomy' => 'project-category', 
						'field'    => 'term_id',
						'terms'    => $selected_category_ids,
					],
				];
			}

			// Get projects based on the query arguments
			$projects = get_posts($query_args);
			$options  = [];

			// Loop through projects and build options array
			foreach ($projects as $project) {
				$options[$project->ID] = get_the_title($project->ID);
			}

			return $options;
		}






		/**
		 * Return taxonomy name with link.
		 *
		 * @since 1.0.0
		 *
		 * @param string $taxonomy . give your taxonomy.
		 * 
		 * @param string $icon_class . give your icon class here.
		 * 
		 * @return mixed return taxonomy name with link.
		 */
		public static function term_with_link($icon_class, $taxonomy)
		{

			$terms = wp_get_object_terms(get_the_ID(), $taxonomy);
			if ($terms ?? ''):

				foreach ((array) $terms as $term): ?>
					<a href="<?php echo get_term_link($term->slug, $taxonomy) ?>"><i class="<?php echo $icon_class ?>"></i>
						<?php echo $term->name ?>
					</a>
				<?php endforeach;

			endif;
		}

		/**
		 * Return taxonomy name with link.
		 *
		 * @since 1.0.0
		 *
		 * @param string $taxonomy . give your taxonomy.
		 * 
		 * @param string $icon_class . give your icon class here.
		 * 
		 * @return mixed return taxonomy name with link.
		 */
		public static function term_without_link($icon_class, $taxonomy)
		{

			$terms = wp_get_object_terms(get_the_ID(), $taxonomy);
			if ($terms ?? ''):
				?>

				<span><i class="<?php echo $icon_class ?>"></i>
					<?php
					foreach ((array) $terms as $term):
						echo $term->name;
					endforeach;
					?>
				</span>
			<?php
			endif;
		}

		/**
		 * Return term link value.
		 *
		 * @since 1.0.0
		 * 
		 * @return mixed Post type option value.
		 */
		public static function get_any_term_link($taxonomy)
		{
			$term = get_the_terms(get_the_ID(), $taxonomy);
			$link = get_term_link($term[0]->slug, $taxonomy);
			return $link;
		}

	
		/**
		 * clean special chars, spaces with hyphens
		 *
		 * @since   1.0.0
		 */
		public static function clean($string)
		{
			$string = str_replace(' ', '', $string);                  // Replaces all spaces with hyphens.
			$string = preg_replace('/[^A-Za-z0-9\-]/', '', $string);  // Removes special chars.

			return preg_replace('/-+/', '', $string);  // Replaces multiple hyphens with single one.
		}
		/**
		 * Return Elementor header footer plugin post list
		 *
		 * @return data
		 */
		public static function get_header_footer()
		{
			$args = array(
				'post_type'   => 'footer_blocks',
				'order'       => 'asc',
				'numberposts' => 99,
			);

			$latest_books = get_posts($args);
			$array        = [];
			foreach ($latest_books as $value) {
				$array[$value->post_name] = $value->post_title;
			}
			return $array;
		}

		/**
		 * Return Elementor header footer post ID or default footer area
		 *
		 * @param string $slug The slug of the header/footer.
		 * @return string The shortcode for the header/footer or default footer area.
		 */
		public static function get_header_footer_shortcode($slug)
		{
			$post = get_page_by_path($slug, OBJECT, 'elementor-hf');
			if ($post) {
				// If header/footer post found, return shortcode
				return '[hfe_template id=' . $post->ID . ']';
			} else {
				// If header/footer post not found, return default footer area
				return self::default_footer_area_sec();
			}
		}


		/**
		 * calculating reading times
		 *
		 * @return void
		 */
		public static function calculate_reading_time($content)
		{
			// Count the number of words in the content.
			$word_count = str_word_count(strip_tags($content));
			// Minimum reading time is 1 minute.
			$reading_time = max(1, round($word_count / 100));
			return $reading_time;
		}


		/**
		 * Get post tags for select
		 *
		 * @return array
		 */
		public static function get_tags_for_select()
		{
			$tags    = get_tags();
			$options = [];
			foreach ($tags as $tag) {
				$options[$tag->term_id] = $tag->name;
			}
			return $options;
		}


		/**
		 * filtering posts by authors
		 *
		 * @return void
		 */
		public static function get_blog_authors()
		{
			// Define an array of roles you want to include
			$roles_to_include = ['author', 'administrator', 'subscriber', 'contributor', 'editor'];

			// Retrieve users based on the defined roles
			$users          = get_users(['role__in' => $roles_to_include]);
			$author_options = ['all' => esc_html__('All Authors', 'vernex-core')];

			foreach ($users as $user) {
				$author_options[$user->ID] = $user->display_name;
			}

			return $author_options;
		}


		/**
		 * get post categories for select
		 *
		 * @return void 
		 */

		public static function get_categories_for_select()
		{
			$categories = get_categories();
			$options    = [];
			foreach ($categories as $category) {
				$options[$category->term_id] = $category->name;
			}
			return $options;
		}



		/**
		 * Generate the HTML for the default footer area
		 *
		 * @return string The HTML for the default footer area.
		 */
		public static function default_footer_area_sec()
		{
			ob_start();  // Start output buffering
			?>
			<div class="container">
				<div class="footer-bottom d-block text-center border-top-0">
					<div class="copyright-area">
						<p><?php echo esc_html__('©Copyright 2025', 'vernex-core') ?>
							<a href="<?php echo esc_url('https://www.poetic-wp.egenslab.com/') ?>">
								<?php echo esc_html__(' poetic', 'vernex-core') ?>
							</a>
							<?php echo esc_html__('| Design By', 'vernex-core') ?>
							<a href="<?php echo esc_url('https://www.egenslab.com/') ?>">
								<?php echo esc_html__(' Egens Lab', 'vernex-core') ?>
							</a>
						</p>
					</div>
				</div>
			</div>
<?php
			$output = ob_get_clean();  // Get and clean the buffered output
			return $output;
		}



		/**
		 * @return [string] video url for video post
		 */
		public static function egns_embeded_video($width = '', $height = '')
		{
			$url = esc_url(get_post_meta(get_the_ID(), 'egns_video_url', 1));
			if (!empty($url)) {
				return wp_oembed_get($url, array('width' => $width, 'height' => $height));
			}
		}


		/**
		 * @return [string] Has embed video for video post.
		 */
		public static function has_egns_embeded_video()
		{
			$url = esc_url(get_post_meta(get_the_ID(), 'egns_video_url', 1));
			if (!empty($url)) {
				return true;
			} else {
				return false;
			}
		}


		/**
		 * 
		 * @return [string] audio url for audio post
		 */
		public static function egns_embeded_audio($width, $height)
		{
			$url = esc_url(get_post_meta(get_the_ID(), 'egns_audio_url', 1));
			if (!empty($url)) {
				return '<div class="post-media">' . wp_oembed_get($url, array('width' => $width, 'height' => $height)) . '</div>';
			}
		}

		/**
		 * @return [string] Checks For Embed Audio In The Post.
		 */
		public static function egns_has_embeded_audio()
		{
			$url = esc_url(get_post_meta(get_the_ID(), 'egns_audio_url', 1));
			if (!empty($url)) {
				return true;
			} else {
				return false;
			}
		}

		/**
		 * Check if the podcast is enabled for the post.
		 *
		 * @return bool True if podcast is enabled, false otherwise.
		 */
		public static function egns_is_podcast_enabled()
		{
			return (bool) get_post_meta(get_the_ID(), 'egns_podcast', true);
		}

		/**
		 * Get the podcast audio URL.
		 *
		 * @return string The podcast audio URL.
		 */
		public static function egns_get_podcast_audio_url()
		{
			return esc_url(get_post_meta(get_the_ID(), 'egns_podcast_audio_url', true));
		}

		/**
		 * Get the podcast audio URL.
		 *
		 * @return string The podcast audio URL.
		 */
		public static function egns_get_podcast_audio__main_url()
		{
			return esc_url(get_post_meta(get_the_ID(), 'egns_audio_url', true));
		}


		/**
		 * Get the podcast video URL.
		 *
		 * @return string The podcast audio URL.
		 */
		public static function egns_get_podcast_video_url()
		{
			return esc_url(get_post_meta(get_the_ID(), 'egns_video_url', true));
		}



		/**
		 * Get the podcast platform list.
		 *
		 * @return array The list of podcast platforms.
		 */
		public static function egns_get_podcast_platform_list()
		{
			$platforms = get_post_meta(get_the_ID(), 'egns_podcast_platform', true);
			return is_array($platforms) ? $platforms : array();
		}



		/**
		 * @return [string] Embed gallery for the post.
		 */
		public static function egns_gallery_images()
		{
			$images = get_post_meta(get_the_ID(), 'egns_gallery_images', 1);

			$images = explode(',', $images);
			if ($images && count($images) > 1) {
				$gallery_slide  = '<div class="swiper blog-archive-slider">';
				$gallery_slide .= '<div class="swiper-wrapper">';
				foreach ($images as $image) {
					$gallery_slide .= '<div class="swiper-slide"><a href="' . get_the_permalink() . '"><img src="' . wp_get_attachment_image_url($image, 'full') . '" alt="' . esc_attr(get_the_title()) . '"></a></div>';
				}
				$gallery_slide .= '</div>';
				$gallery_slide .= '</div>';

				$gallery_slide .= '<div class="slider-arrows d-flex flex-row justify-content-between align-items-center">';
				$gallery_slide .= '<div class="blog1-prev swiper-prev-arrow" tabindex="0" role="button" aria-label="' . esc_html('Previous slide') . '"> <i class="bi bi-arrow-left"></i> </div>';

				$gallery_slide .= '<div class="blog1-next swiper-next-arrow" tabindex="0" role="button" aria-label="' . esc_html('Next slide') . '"><i class="bi bi-arrow-right"></i></div>';
				$gallery_slide .= '</div>';

				return $gallery_slide;
			}
		}

		/**
		 * @return [string] Has Gallery for Gallery post.
		 */
		public static function has_egns_gallery()
		{
			$images = get_post_meta(get_the_ID(), 'egns_gallery_images', 1);
			if (!empty($images)) {
				return true;
			} else {
				return false;
			}
		}


		/**
		 * @return string get the attachment image.
		 */
		public static function egns_thumb_image()
		{
			$image = get_post_meta(get_the_ID(), 'egns_thumb_images', 1);
			echo '<a href="' . get_the_permalink() . '"><img src="' . $image['url'] . '" alt="' . esc_attr("image") . ' "class="img-fluid wp-post-image"></a>';
		}

		/**
		 * @return [quote] text for quote post
		 */
		public static function egns_quote_content()
		{
			$text = get_post_meta(get_the_ID(), 'egns_quote_text', 1);
			if (!empty($text)) {
				return sprintf(esc_attr__("%s", 'vernex-core'), $text);
			}
		}
	} //End Main Class


}//end if
