document.addEventListener('DOMContentLoaded', function() {
    // Mobile menu elements
    const mobileToggle = document.querySelector('.mobile-menu-toggle');
    const mobileNav = document.querySelector('.mobile-nav');
    const mobileOverlay = document.querySelector('.mobile-nav-overlay');
    const mobileCloseBtn = document.querySelector('.mobile-close-btn');
    
    // Mobile submenu elements
    const mobileNavHeaders = document.querySelectorAll('.mobile-nav-header-item');
    const mobileBackBtns = document.querySelectorAll('.mobile-back-btn');
    
    // Desktop nav links for active state
    const navLinks = document.querySelectorAll('.nav-link');

    // Set active page based on current URL or simulate active state
    function setActivePage() {
        // Remove active class from all links
        navLinks.forEach(link => link.classList.remove('active'));
        
        // For demo purposes, set "Our Services" as active
        // In real implementation, you'd check window.location.pathname
        const activeLink = document.querySelector('[data-page="services"]');
        if (activeLink) {
            activeLink.classList.add('active');
        }
    }

    // Initialize active page
    setActivePage();

    // Handle nav link clicks for active state
    navLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            
            // Remove active class from all links
            navLinks.forEach(l => l.classList.remove('active'));
            
            // Add active class to clicked link
            this.classList.add('active');
        });
    });

    // Mobile menu toggle
    function openMobileMenu() {
        mobileToggle.classList.add('active');
        mobileNav.classList.add('active');
        mobileOverlay.classList.add('active');
        document.body.style.overflow = 'hidden';
    }

    function closeMobileMenu() {
        mobileToggle.classList.remove('active');
        mobileNav.classList.remove('active');
        mobileOverlay.classList.remove('active');
        document.body.style.overflow = '';
        
        // Close any open submenus
        document.querySelectorAll('.mobile-submenu').forEach(submenu => {
            submenu.classList.remove('active');
        });
    }

    mobileToggle.addEventListener('click', openMobileMenu);
    mobileCloseBtn.addEventListener('click', closeMobileMenu);
    mobileOverlay.addEventListener('click', closeMobileMenu);

    // Mobile submenu navigation (slide from right to left)
    mobileNavHeaders.forEach(header => {
        header.addEventListener('click', function() {
            const navItem = this.parentElement;
            const submenu = navItem.querySelector('.mobile-submenu');
            
            if (submenu) {
                submenu.classList.add('active');
            }
        });
    });

    // Mobile back button functionality
    mobileBackBtns.forEach(backBtn => {
        backBtn.addEventListener('click', function() {
            const submenu = this.closest('.mobile-submenu');
            submenu.classList.remove('active');
        });
    });

    // Close mobile menu when clicking on links
    const mobileLinks = document.querySelectorAll('.mobile-nav a:not(.mobile-back-btn)');
    mobileLinks.forEach(link => {
        link.addEventListener('click', function() {
            closeMobileMenu();
        });
    });

    // Desktop mega menu management - Fix for multiple menus opening
    const megaMenuItems = document.querySelectorAll('.mega-menu-item');
    let activeMenu = null;
    let hoverTimeout = null;

    megaMenuItems.forEach(item => {
        const megaMenu = item.querySelector('.mega-menu');
        
        item.addEventListener('mouseenter', function() {
            // Clear any existing timeout
            if (hoverTimeout) {
                clearTimeout(hoverTimeout);
                hoverTimeout = null;
            }
            
            // Close any currently active menu
            if (activeMenu && activeMenu !== item) {
                const activeMegaMenu = activeMenu.querySelector('.mega-menu');
                activeMegaMenu.style.opacity = '0';
                activeMegaMenu.style.visibility = 'hidden';
                activeMegaMenu.style.transform = 'translateX(-50%) translateY(-10px)';
            }
            
            // Open current menu
            activeMenu = item;
            megaMenu.style.opacity = '1';
            megaMenu.style.visibility = 'visible';
            megaMenu.style.transform = 'translateX(-50%) translateY(0)';
        });
        
        item.addEventListener('mouseleave', function() {
            // Set timeout to close menu with delay
            hoverTimeout = setTimeout(() => {
                if (activeMenu === item) {
                    megaMenu.style.opacity = '0';
                    megaMenu.style.visibility = 'hidden';
                    megaMenu.style.transform = 'translateX(-50%) translateY(-10px)';
                    activeMenu = null;
                }
            }, 150);
        });

        // Keep menu open when hovering over the mega menu itself
        megaMenu.addEventListener('mouseenter', function() {
            if (hoverTimeout) {
                clearTimeout(hoverTimeout);
                hoverTimeout = null;
            }
        });

        megaMenu.addEventListener('mouseleave', function() {
            hoverTimeout = setTimeout(() => {
                megaMenu.style.opacity = '0';
                megaMenu.style.visibility = 'hidden';
                megaMenu.style.transform = 'translateX(-50%) translateY(-10px)';
                activeMenu = null;
            }, 150);
        });
    });

    // Close all mega menus when clicking outside
    document.addEventListener('click', function(e) {
        const isInsideNav = e.target.closest('.nav-menu');
        if (!isInsideNav && activeMenu) {
            const activeMegaMenu = activeMenu.querySelector('.mega-menu');
            activeMegaMenu.style.opacity = '0';
            activeMegaMenu.style.visibility = 'hidden';
            activeMegaMenu.style.transform = 'translateX(-50%) translateY(-10px)';
            activeMenu = null;
        }
    });

    // Keyboard navigation support
    document.addEventListener('keydown', function(e) {
        // Close mobile menu with Escape key
        if (e.key === 'Escape') {
            if (mobileNav.classList.contains('active')) {
                closeMobileMenu();
            }
            
            // Close mega menus with Escape key
            if (activeMenu) {
                const activeMegaMenu = activeMenu.querySelector('.mega-menu');
                activeMegaMenu.style.opacity = '0';
                activeMegaMenu.style.visibility = 'hidden';
                activeMegaMenu.style.transform = 'translateX(-50%) translateY(-10px)';
                activeMenu = null;
            }
        }
    });

    // Resize handler
    const handleResize = debounce(function() {
        // Close mobile menu on resize to desktop
        if (window.innerWidth > 768) {
            closeMobileMenu();
        }
        
        // Close mega menus on resize
        if (activeMenu) {
            const activeMegaMenu = activeMenu.querySelector('.mega-menu');
            activeMegaMenu.style.opacity = '0';
            activeMegaMenu.style.visibility = 'hidden';
            activeMegaMenu.style.transform = 'translateX(-50%) translateY(-10px)';
            activeMenu = null;
        }
    }, 250);

    window.addEventListener('resize', handleResize);

    // Smooth scroll for anchor links
    const anchorLinks = document.querySelectorAll('a[href^="#"]');
    anchorLinks.forEach(link => {
        link.addEventListener('click', function(e) {
            e.preventDefault();
            const target = document.querySelector(this.getAttribute('href'));
            if (target) {
                target.scrollIntoView({
                    behavior: 'smooth',
                    block: 'start'
                });
            }
        });
    });
});

// Utility functions
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}